%=========================================================================
%
%  "Time Preference and the Distributions of Wealth and Income"
%   By Richard M. H. Suen 
%   Economic Inquiry (forthcoming)
%
%   This program is used to generate the benchmark results reported in 
%   Table 4 of the paper. 
%
%=========================================================================

close all
clear all

pause on

%=======================================
%
%  Assign parameter values.
%
%=======================================

alpha   = 0.33;

nu      = [1.0          1.2          1.5         2.0          1.0          1.2          1.5          2.0];
theta   = [0.01202      0.01202      0.01202     0.01202      0.01202      0.01408      0.01694      0.02105];
delta_k = [0.080044917  0.080005107  0.079965515 0.079927189  0.080044917  0.080132753  0.080266828  0.08046976];

sigma = 1;
gamma = 1.022;

N = 1000;
beta_max = 0.992;
beta_min = 0.966;

beta = linspace(beta_min, beta_max, N);

eps  = 0.939; 
vsig = 0.871;
vphi = 1;

delta_h = 0.037; 
delta_k_hat = gamma - 1 + delta_k; 

r_bar = ( gamma/beta_max + theta*gamma )/(1 + theta) - (1 - delta_k);


%==============================================================================
%
%  Distributions of labor hours, human capital and wealth
%
%==============================================================================

r_eq = 0.11;
Kd = (alpha/r_eq)^(1/(1 - alpha));
K_Y_Ratio = Kd^(1 - alpha);

w = (1 - alpha)*(Kd^alpha);


%=============================================================================

M = length(nu); 

Results = zeros(M, 16);
Errors  = zeros(M, 1);

lambda  = zeros(1, N);

lambda_l  = zeros(1,N);
lambda_e  = zeros(1,N);
lambda_k  = zeros(1,N);
lambda_in = zeros(1,N);

share_pop = zeros(M,N+1);
share_k   = zeros(M,N+1);

for i = 1:M
       
    lambda(1) = (1/N)^(1/nu(i));
    
    for j = 2:N
        
        lambda(j) = (j/N)^(1/nu(i)) - ((j-1)/N)^(1/nu(i));
        
    end
    
    P = (((gamma^(sigma - 1))./beta - (1 - delta_h))/delta_h - vsig )/eps;
    
    l = P./(1 + P);
    h = ( ( vphi/delta_h )*(1 - l).^eps ).^(1/(1 - vsig));
    earn = w*(l.*h);
    
    D = ((( gamma^sigma )./beta - (1 - delta_k(i)) - r_eq )/theta(i)).^(1/sigma);
    
    k = earn./( D - r_eq + delta_k_hat(i) );
    
    if ( sum(k > 0) < N)
        
        fprintf('Error: Capital cannot be negative!')
        [nu(i) theta(i)]
        
    end
    
    c = D.*k;
    
    inc = earn + r_eq*k;
    
    Res = earn + (r_eq - delta_k_hat(i))*k - c;
    Check = sum(Res);
    Errors(i) = Kd - (lambda*k')/(lambda*(l.*h)');  % Errors in the capital market clearing condition.
    
    %=================================================================================
    % 
    %  Summary Statistics for the Distributions
    %
    %=================================================================================

    
    [l2,   ix1] = sort(l);
    [e2,   ix2] = sort(earn); 
    [k2,   ix3] = sort(k);
    [inc2, ix4] = sort(inc);
    
    
    for j = 1:N
        
        lambda_l(j)  = lambda(ix1(j));
        lambda_e(j)  = lambda(ix2(j));
        lambda_k(j)  = lambda(ix3(j));
        lambda_in(j) = lambda(ix4(j)); 
        
    end
    
    s1 = (lambda_l.*l2)/(lambda_l*l2');
    s2 = (lambda_e.*e2)/(lambda_e*e2');
    s3 = (lambda_k.*k2)/(lambda_k*k2');
    s4 = (lambda_in.*inc2)/(lambda_in*inc2');
    
    share_pop(i,:) = [0 cumsum(lambda_k)];
    share_l        = [0 cumsum(s1)];
    share_e        = [0 cumsum(s2)];
    share_k(i,:)   = [0 cumsum(s3)];
    share_in       = [0 cumsum(s4)];
    
    Area_l  = 0;
    Area_e  = 0;
    Area_k  = 0;
    Area_in = 0;
    
    
    for j = 1:N
        
        Area_l  = Area_l  + ( share_l(j+1)    + share_l(j) )*lambda_l(j)/2;
        Area_e  = Area_e  + ( share_e(j+1)    + share_e(j) )*lambda_e(j)/2;
        Area_k  = Area_k  + ( share_k(i,j+1)  + share_k(i,j) )*lambda_k(j)/2;
        Area_in = Area_in + ( share_in(j+1)   + share_in(j) )*lambda_in(j)/2;
        
    end
    
    Gini_l  = 1 - Area_l/0.5;
    Gini_e  = 1 - Area_e/0.5;
    Gini_in = 1 - Area_in/0.5;
    Gini_k  = 1 - Area_k/0.5;
    
    mean_l = lambda*l';
    var_l  = lambda*(( l' - mean_l ).^2);
    std_l  = sqrt(var_l);
    CV_l   = std_l/mean_l;
    
    mean_e = lambda*earn';
    var_e  = lambda*(( earn' - mean_e ).^2);
    std_e  = sqrt(var_e);
    CV_e   = std_e/mean_e;
    
    mean_k = lambda*k';
    var_k  = lambda*(( k' - mean_k ).^2);
    std_k  = sqrt(var_k);
    CV_k   = std_k/mean_k;
    
    mean_in = lambda*inc';
    var_in  = lambda*(( inc' - mean_in ).^2);
    std_in  = sqrt(var_in);
    CV_in   = std_in/mean_in;
    
    S40 = share_k(i,400);
    S90 = 1 - share_k(i,901);
    S95 = 1 - share_k(i,951);
    S99 = 1 - share_k(i,991);
    
    T40 = share_in(400);
    T90 = 1 - share_in(901);
    T95 = 1 - share_in(951);
    T99 = 1 - share_in(991);
    
    Source = (earn)./(earn + r_eq*k);
    L1 = mean(Source(1:10));
    L2 = mean(Source(11:50));
    L3 = mean(Source(51:100));
    L4 = mean(Source(401:600));
    L5 = mean(Source(901:950));
    L6 = mean(Source(951:990));
    L7 = mean(Source(991:1000));
    
    mean_beta = lambda*beta';
    var_beta  = lambda*( ( beta' - mean_beta ).^2 );
    std_beta  = sqrt(var_beta);
    skew_beta = lambda*( ( beta' - mean_beta ).^3 );
    
    x = ((0.5)^nu(i))*N;
    beta_floor = beta(floor(x));
    beta_ceil  = beta(ceil(x));
    
    median_beta = (beta_floor + beta_ceil)/2;
    
    Results(i,:) = [nu(i) mean_beta median_beta lambda(1) lambda(end) Gini_e Gini_in Gini_k T40 T90 T95 T99 S40 S90 S95 S99];

    
end

%==============================================
%  Results reported in Table 4
%==============================================

Results





